﻿package away3d.core.filter
{
    import away3d.cameras.*;
    import away3d.containers.*;
    import away3d.core.base.Object3D;
    import away3d.core.clip.*;
    import away3d.core.draw.*;
    import away3d.core.render.*;
    import away3d.core.utils.*;
    import away3d.materials.*;

    /**
    * Adds fog layers to a view and provides automatic farfield filtering for primitives outside the furthest fog layers.
	*/
    public class FogFilter implements IPrimitiveFilter
    {
    	private var i:int;
    	
    	private var _primitives:Array;
    	private var pri:DrawPrimitive;
    	
    	private var _material:IFogMaterial;
    	private var _minZ:Number;
    	private var _maxZ:int;
    	private var _subdivisions:int;
    	private var _materials:Array = [];
    	
    	private var _fogPrimitives:Array = [];
    	private var fog:DrawFog;
		 
        /**
        * Instance of the Init object used to hold and parse default property values
        * specified by the initialiser object in the 3d object constructor.
        */
		protected var ini:Init;
		
		/**
		 * Defines the material used by the fog layers.
		 */
		public function get material():IFogMaterial
		{
			return _material;
		}
		
		public function set material(val:IFogMaterial):void
		{
			_material = val;
		}
		
		/**
		 * Allows color change at runtime of the filter
		 * @param	color			The new color for the filter
		 */
		public function updateMaterialColor(color:uint):void
		{
			for each (var fog:DrawFog in _fogPrimitives) {
				if(fog.material is ColorMaterial)
					fog.material = new ColorMaterial(color, {alpha:fog.material.alpha});
			}
		}
    	
		/**
		 * Creates a new <code>FogFilter</code> object.
		 *
		 * @param	init			[optional]	An initialisation object for specifying default instance properties.
		 */
		function FogFilter(init:Object = null):void
		{
			ini = Init.parse(init);
			
			_material = ini.getMaterial("material") as IFogMaterial;
			_minZ = ini.getNumber("minZ", 1000, {min:0});
            _maxZ = ini.getNumber("maxZ", 5000, {min:0});
            _subdivisions = ini.getInt("subdivisions", 20, {min:1, max:50});
            _materials = ini.getArray("materials");
            
            if (!(_material is IFogMaterial))
            	throw new Error("FogFilter requires IFogMaterial");
            
            if (!_material && !_materials.length)
            	_material = new ColorMaterial(0x000000);
            
            //materials override subdivisions
            if (!_materials.length) {
            	i = _subdivisions;
            	while (i--)
            		_materials.push(_material.clone());
            } else {
            	_subdivisions = _materials.length;
            }
            
            i = _subdivisions;
            while(i--) {
            	(_materials[i] as IFogMaterial).alpha = 0.45*i/_subdivisions;
            	fog = new DrawFog();
            	fog.screenZ = _minZ + (_maxZ - _minZ)*i/(_subdivisions - 1);
            	fog.material = _materials[i];
            	_fogPrimitives.unshift(fog);
            }
		}
        
		/**
		 * @inheritDoc
		 */
        public function filter(primitives:Array, scene:Scene3D, camera:Camera3D, clip:Clipping):Array
        {
        	if (!primitives.length || !primitives[0].source || primitives[0].source.session != scene.session)
        		return primitives;
        	
			for each (fog in _fogPrimitives) {
				fog.source = scene;
				fog.clip = clip;
				primitives.push(fog);
			}
			
			_primitives = [];
			
			for each (pri in primitives) {
				if (pri.screenZ < _maxZ)
					_primitives.push(pri); 
			}

        	return _primitives;
        }
		
		/**
		 * Used to trace the values of a filter.
		 * 
		 * @return A string representation of the filter object.
		 */
        public function toString():String
        {
            return "FogFilter";
        }
    }
}
